﻿<?php
// ============================================================================
// ARQUIVO: salvar_notas.php
// Função: processar e salvar as notas lançadas pelo professor
// ============================================================================

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once "conexao.php"; // Conexão com o banco de dados

// Variável auxiliar para normalização/validação (aceita ',' ou '.')
// A função garante que o valor seja float (com ponto) e esteja entre 0 e 20
$normalizar_nota = function ($valor) {
    if ($valor === '' || $valor === null) {
        return null;
    }
    // 1. Converte vírgula para ponto.
    $valor_limpo = str_replace(',', '.', $valor);
    // 2. Tenta converter para float.
    $nota = (float)$valor_limpo;
    
    // 3. Valida o intervalo.
    if ($nota < 0 || $nota > 20) {
        return null; 
    }
    return $nota;
};

// Bloco principal de processamento POST
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['salvar_notas'])) {

    $turma_id = (int)($_POST['turma_id'] ?? 0);
    $disciplina_id = (int)($_POST['disciplina_id'] ?? 0);
    $trimestre_selecionado = (int)($_POST['trimestre'] ?? 0);
    $aluno_ids = $_POST['aluno_id'] ?? [];

    if ($turma_id > 0 && $disciplina_id > 0 && $trimestre_selecionado > 0 && !empty($aluno_ids)) {

        $conn->begin_transaction();
        $sucesso_salvamento = true;

        foreach ($aluno_ids as $aluno_id) {
            $aluno_id = (int)$aluno_id;

            // --- 1. CAPTURA DOS NOVOS NOMES DE CAMPOS (NPP E NPT) ---
            $npp = $normalizar_nota($_POST["npp_{$trimestre_selecionado}_{$aluno_id}"] ?? null);
            $npt = $normalizar_nota($_POST["npt_{$trimestre_selecionado}_{$aluno_id}"] ?? null);

            // --- 2. CAPTURA DOS NOVOS NOMES DE CAMPOS (NEO E NEE) - APENAS NO 3º TRIMESTRE ---
            $neo = null;
            $nee = null;
            if ($trimestre_selecionado == 3) {
                $neo = $normalizar_nota($_POST["neo_{$aluno_id}"] ?? null);
                $nee = $normalizar_nota($_POST["nee_{$aluno_id}"] ?? null);
            }

            // --- 3. CÁLCULO DE MT ---
            $mt = ($npp !== null && $npt !== null) ? $normalizar_nota(($npp + $npt) / 2) : null;
            
            // --- 4. CAPTURA DA OBSERVAÇÃO ---
            $observacao = trim($_POST["obs_{$aluno_id}"] ?? '');
            
            // --- 5. SALVAMENTO NO BANCO (INSERT/UPDATE) ---
            $stmt = $conn->prepare("
                INSERT INTO notas (aluno_id, disciplina_id, trimestre, npp, npt, mt, neo, nee, observacao, data_lancamento)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
                ON DUPLICATE KEY UPDATE 
                    npp = VALUES(npp), 
                    npt = VALUES(npt), 
                    mt = VALUES(mt),
                    neo = VALUES(neo),
                    nee = VALUES(nee),
                    observacao = VALUES(observacao), 
                    data_lancamento = NOW()
            ");

            // Os tipos dos parâmetros devem ser 'i' (int) ou 'd' (double/float) para notas, 's' (string) para observação
            $stmt->bind_param("iiddiddds", 
                $aluno_id, 
                $disciplina_id, 
                $trimestre_selecionado, 
                $npp, 
                $npt, 
                $mt,
                $neo,
                $nee,
                $observacao
            );

            if (!$stmt->execute()) {
                $sucesso_salvamento = false;
                error_log("Erro ao salvar nota do aluno $aluno_id: " . $stmt->error);
                break;
            }
            $stmt->close();
        }

        // Finalização da transação
        if ($sucesso_salvamento) {
            $conn->commit();
            $_SESSION['mensagem_sucesso'] = "Notas salvas com sucesso!";
        } else {
            $conn->rollback();
            $_SESSION['mensagem_erro'] = "Erro ao salvar uma ou mais notas.";
        }

        // Redirecionamento para evitar reenvio do formulário
        $destino = "mini_pauta_professor1.php?turma_id=$turma_id&disciplina_id=$disciplina_id&trimestre=$trimestre_selecionado";
        header("Location: " . $destino);
        exit;
        
    } else {
        $_SESSION['mensagem_erro'] = "Dados incompletos para salvar.";
        header("Location: mini_pauta_professor1.php");
        exit;
    }
}
?>